function results = imm(X,alpha,M)
    
    %Particle filtering for infinite mixture model of conditioning.
    %
    %USAGE: results = imm(X,[alpha],[M])
    %
    %INPUTS:
    %   X - [T x D] stimulus inputs, where T is the number of timepoints
    %       and D is the number of stimulus features. We assume features
    %       are binary here.
    %   alpha (optional) - concentration parameter (default: 0.1)
    %   M (optional) - number of samples (default: 100)
    %
    %OUTPUTS:
    %   results - structure wih the following fields:
    %       .opts - options (missing fields set to defaults)
    %       .V - [T x 1] US prediction
    %       .Z - [T x T] latent cause posterior, where Z(t,k) is the
    %            probability of latent cause k being active on trial t
    %
    %Sam Gershman, October 2011
    
    %set defaults
    if nargin < 2 || isempty(alpha); alpha = 0.1; end
    if nargin < 3 || isempty(M); M = 100; end
    
    %initialization
    [T D] = size(X);
    N = zeros(M,T,D);           %feature-cluster co-occurence counts
    B = zeros(M,T,D);           %feature-cluster co-occurence counts
    Nk = zeros(M,T);            %cluster counts
    results.Z = zeros(T,T);     %cluster assignments
    results.V = zeros(T,1);     %US predictions
    
    %loop over trials
    for t = 1:T
        
        %calculate CRP prior
        prior = Nk;
        for m = 1:M
            prior(m,find(prior(m,:)==0,1)) = alpha;
        end
        prior = bsxfun(@rdivide,prior,sum(prior,2));
        
        %calculate likelihood
        lik = N;

        lik(:,:,X(t,:)==0) = B(:,:,X(t,:)==0);
        lik = bsxfun(@rdivide,lik+1,Nk+2);
        
        %US prediction
        post = prior.*squeeze(prod(lik(:,:,2:D),3));
        post0 = bsxfun(@rdivide,post,sum(post,2));
        nUS = squeeze(N(:,:,1)+1)./(Nk+2);
%         results.nUS(t,:,:) = nUS;
%         results.post(t,:,:) = post0;
        results.V(t,1) = post0(:)'*nUS(:)./M;
        
        %sample cluster assignment and update sufficient statistics
        post = post.*squeeze(lik(:,:,1));
        post = bsxfun(@rdivide,post,sum(post,2));
        post = mean(post);
        results.Z(t,:) = post;
        for m = 1:M
            [~, k] = histc(rand,[0 cumsum(post)]); %multinomial sample
            Nk(m,k) = Nk(m,k) + 1;
            N(m,k,X(t,:)==1) = N(m,k,X(t,:)==1) + 1;
            B(m,k,X(t,:)==0) = B(m,k,X(t,:)==0) + 1;
        end  
    end